<?php
/**
 * SerializedDataHandler - Sichere Handhabung serialisierter Daten
 * 
 * Diese Klasse behebt das Hauptproblem mit Bricks Builder und anderen
 * Page Builders: Korruption serialisierter Daten durch URL-Migration
 * 
 * @package JenvaBackupMigration
 * @since 1.0.1
 */

namespace JenvaBackupMigration\Restore;

if (!defined('ABSPATH')) {
    exit;
}

class SerializedDataHandler {
    
    /** @var array Statistiken */
    private $stats = [
        'processed' => 0,
        'repaired' => 0,
        'failed' => 0,
        'errors' => [],
    ];
    
    /**
     * Sichere URL-Migration in serialisierten Daten
     * 
     * Diese Methode ist der Schlüssel zur Lösung der Bricks-Probleme!
     * 
     * @param string $serialized_data Serialisierte Daten
     * @param string $old_url Alte URL  
     * @param string $new_url Neue URL
     * @return string Migrierte Daten
     */
    public function migrateUrls(string $serialized_data, string $old_url, string $new_url): string {
        $this->stats['processed']++;
        
        // Wenn keine URL enthalten, direkt zurückgeben
        if (strpos($serialized_data, $old_url) === false) {
            return $serialized_data;
        }
        
        // Methode 1: Saubere Deserialisierung (bevorzugt)
        $data = @unserialize($serialized_data);
        
        if ($data !== false) {
            // Sauber deserialisierede Daten - sichere URL-Ersetzung
            $migrated_data = $this->replaceUrlsRecursive($data, $old_url, $new_url);
            return serialize($migrated_data);
        }
        
        // Methode 2: Reparatur korrupter Daten
        if ($this->looksLikeSerialized($serialized_data)) {
            $this->stats['repaired']++;
            
            $repaired = $this->repairSerializedData($serialized_data);
            if ($repaired !== null) {
                $migrated = $this->replaceUrlsRecursive($repaired, $old_url, $new_url);
                return serialize($migrated);
            }
        }
        
        // Methode 3: String-basierte Reparatur (Fallback)
        $result = $this->fallbackStringReplace($serialized_data, $old_url, $new_url);
        
        // Validierung des Ergebnisses
        if (@unserialize($result) !== false) {
            return $result;
        }
        
        // Letzte Möglichkeit: Original mit direkter Ersetzung
        $this->stats['failed']++;
        $this->stats['errors'][] = "Could not safely migrate URLs in serialized data";
        
        return str_replace($old_url, $new_url, $serialized_data);
    }
    
    /**
     * Rekursive URL-Ersetzung in beliebigen Datenstrukturen
     * 
     * @param mixed $data Daten (String, Array, Objekt)
     * @param string $old_url Alte URL
     * @param string $new_url Neue URL
     * @return mixed Migrierte Daten
     */
    private function replaceUrlsRecursive($data, string $old_url, string $new_url) {
        if (is_string($data)) {
            return str_replace($old_url, $new_url, $data);
        }
        
        if (is_array($data)) {
            foreach ($data as $key => $value) {
                $data[$key] = $this->replaceUrlsRecursive($value, $old_url, $new_url);
            }
            return $data;
        }
        
        if (is_object($data)) {
            foreach ($data as $key => $value) {
                $data->$key = $this->replaceUrlsRecursive($value, $old_url, $new_url);
            }
            return $data;
        }
        
        return $data;
    }
    
    /**
     * Prüft ob String wie serialisierte Daten aussieht
     */
    private function looksLikeSerialized(string $data): bool {
        $trimmed = trim($data);
        
        if (empty($trimmed)) return false;
        
        // Typische Serialisierungs-Patterns
        $patterns = [
            '/^a:\d+:\{/',           // Array: a:1:{
            '/^s:\d+:"/',            // String: s:5:"hello"
            '/^O:\d+:"/',            // Object: O:8:"stdClass"
            '/^i:\d+;/',             // Integer: i:123;
            '/^d:[\d\.]+;/',         // Double: d:1.23;
            '/^b:[01];/',            // Boolean: b:1;
            '/^N;/',                 // Null: N;
        ];
        
        foreach ($patterns as $pattern) {
            if (preg_match($pattern, $trimmed)) {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Versucht korrupte serialisierte Daten zu reparieren
     */
    private function repairSerializedData(string $data): ?array {
        // Methode 1: String-Längen korrigieren
        $repaired = $this->fixStringLengths($data);
        $test = @unserialize($repaired);
        
        if ($test !== false) {
            return $test;
        }
        
        // Methode 2: Aggressive Reparatur
        return $this->aggressiveRepair($data);
    }
    
    /**
     * Korrigiert String-Längen in serialisierten Daten  
     * 
     * Das ist der Hauptgrund für Bricks-Probleme: falsche Byte-Counts!
     */
    private function fixStringLengths(string $data): string {
        // Pattern: s:NUMBER:"STRING";
        return preg_replace_callback(
            '/s:(\d+):"((?:[^"\\\\]|\\\\.)*)";/s',
            function($matches) {
                $original_length = (int)$matches[1];
                $string = $matches[2];
                $actual_length = strlen($string);
                
                // Nur ersetzen wenn Länge nicht stimmt
                if ($original_length !== $actual_length) {
                    return 's:' . $actual_length . ':"' . $string . '";';
                }
                
                return $matches[0];
            },
            $data
        );
    }
    
    /**
     * Fallback: String-basierte URL-Ersetzung mit Längenkorrektur
     */
    private function fallbackStringReplace(string $data, string $old_url, string $new_url): string {
        // URL ersetzen
        $replaced = str_replace($old_url, $new_url, $data);
        
        // String-Längen korrigieren
        return $this->fixStringLengths($replaced);
    }
    
    /**
     * Aggressive Reparatur für stark korrupte Daten
     */
    private function aggressiveRepair(string $data): ?array {
        // Versuche JSON-like Struktur zu extrahieren und als Array zu interpretieren
        if (strpos($data, 'a:') === 0) {
            // Array-Pattern erkennen
            if (preg_match('/^a:(\d+):\{(.+)\}$/s', $data, $matches)) {
                $count = (int)$matches[1];
                $content = $matches[2];
                
                // Versuche Schlüssel-Wert-Paare zu extrahieren
                $array = $this->parseArrayContent($content);
                
                if (is_array($array) && count($array) <= $count * 2) { // Key-Value pairs
                    return $this->buildAssociativeArray($array);
                }
            }
        }
        
        return null;
    }
    
    /**
     * Parst Array-Inhalt aus serialisierten Daten
     */
    private function parseArrayContent(string $content): ?array {
        $elements = [];
        $pos = 0;
        $length = strlen($content);
        
        while ($pos < $length) {
            // String finden: s:N:"value";
            if (preg_match('/^s:(\d+):"((?:[^"\\\\]|\\\\.)*)";/', substr($content, $pos), $matches)) {
                $elements[] = $matches[2];
                $pos += strlen($matches[0]);
                continue;
            }
            
            // Integer finden: i:N;
            if (preg_match('/^i:(\d+);/', substr($content, $pos), $matches)) {
                $elements[] = (int)$matches[1];
                $pos += strlen($matches[0]);
                continue;
            }
            
            // Unbekanntes Element - verwerfen
            $pos++;
        }
        
        return $elements;
    }
    
    /**
     * Baut assoziatives Array aus geparsten Elementen
     */
    private function buildAssociativeArray(array $elements): array {
        $result = [];
        
        for ($i = 0; $i < count($elements); $i += 2) {
            if (isset($elements[$i + 1])) {
                $key = $elements[$i];
                $value = $elements[$i + 1];
                $result[$key] = $value;
            }
        }
        
        return $result;
    }
    
    /**
     * Gibt Statistiken zurück
     */
    public function getStats(): array {
        return $this->stats;
    }
    
    /**
     * Setzt Statistiken zurück
     */
    public function resetStats(): void {
        $this->stats = [
            'processed' => 0,
            'repaired' => 0,
            'failed' => 0,
            'errors' => [],
        ];
    }
}